using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;
using static SupplementaryAreaPlugin.SupplementaryAreasConstants;

namespace SupplementaryAreaPlugin.Actions;

internal interface IGetSupplementaryContext
{
    IAppAccess? App { get; }
    PluginResponse Ok(object shape);
    PluginResponse Fail(int status, string code, string? detail = null);
    string GetQueryValue(IReadOnlyDictionary<string, string?> query, string key);
}

/// <summary>
/// Handles the primary GET action that returns the supplementary model, record summary,
/// field metadata, and dropdown options.
/// Model now contains all data - no need to extract from SharePoint fields.
/// </summary>
internal sealed class ActionGetSupplementary
{
    private readonly IGetSupplementaryContext ctx;
    private readonly IModelRepository repo;

    public ActionGetSupplementary(IGetSupplementaryContext ctx, IModelRepository repo)
    {
        this.ctx = ctx;
        this.repo = repo;
    }

    /// <summary>
    /// Entry point for the main GET action. Loads model from plugin data and returns
    /// complete state with metadata.
    /// </summary>
    public async Task<PluginResponse> ExecuteAsync(IReadOnlyDictionary<string, string?> query, CancellationToken ct)
    {
        if (ctx.App == null)
            return ctx.Fail(500, ErrorNotInitialised);

        var recordId = ctx.GetQueryValue(query, QueryRecordId);
        if (string.IsNullOrWhiteSpace(recordId))
            return ctx.Fail(400, ErrorMissingRecordId);

        // Load model from plugin data (contains complete state)
        var model = await repo.GetOrCreateAsync(recordId, ct).ConfigureAwait(false);

        var recordSummary = await FetchRecordSummaryAsync(recordId, ct).ConfigureAwait(false);
        var activeFields = await FetchActiveContentTypeFields(recordSummary, ct).ConfigureAwait(false);
        var requiredFieldsFromModel = BuildRequiredFieldsList();
        var missingFields = IdentifyMissingFields(requiredFieldsFromModel, activeFields);
        var dropdownOptions = await BuildDropdownOptionsAsync(ct).ConfigureAwait(false);
        
        var responseData = BuildGetResponse(model, recordSummary, activeFields, requiredFieldsFromModel, missingFields, dropdownOptions);
        return ctx.Ok(responseData);
    }

    private async Task<RecordSummary?> FetchRecordSummaryAsync(string recordId, CancellationToken ct)
    {
        var app = ctx.App;
        if (app == null) return null;
        try
        {
            return await app.Records.GetRecordAsync(recordId, ct).ConfigureAwait(false);
        }
        catch (OperationCanceledException) { throw; }
        catch (ArgumentException ex) { app.Logger.LogWarning(ex, "Invalid record ID format '{RecordId}'", recordId); return null; }
        catch (UnauthorizedAccessException ex) { app.Logger.LogWarning(ex, "Access denied for {RecordId}", recordId); return null; }
        catch (Exception ex) { app.Logger.LogError(ex, "Unexpected error fetching record {RecordId}", recordId); return null; }
    }

    private async Task<List<SupplementaryAreasPlugin.FieldDescriptor>> FetchActiveContentTypeFields(RecordSummary? recordSummary, CancellationToken ct)
    {
        var list = new List<SupplementaryAreasPlugin.FieldDescriptor>();
        var app = ctx.App;
        if (recordSummary is null || app is null) return list;
        try
        {
            string parameters = JsonSerializer.Serialize(new { contentType = recordSummary.Type });
            string json = await app.UniversalAppAccess.InvokeActionAsync(UniversalActionGetContentTypeFields, parameters, ct).ConfigureAwait(false);
            using var doc = JsonDocument.Parse(json);
            if (doc.RootElement.ValueKind == JsonValueKind.Array)
            {
                foreach (var el in doc.RootElement.EnumerateArray())
                {
                    ct.ThrowIfCancellationRequested();
                    if (!el.TryGetProperty("internalName", out var internalNameEl)) continue;
                    string? internalName = internalNameEl.ValueKind switch
                    {
                        JsonValueKind.String => internalNameEl.GetString(),
                        JsonValueKind.Number => internalNameEl.GetRawText(),
                        _ => null
                    };
                    if (string.IsNullOrWhiteSpace(internalName)) continue;
                    string? displayName = el.TryGetProperty("displayName", out var dn) && dn.ValueKind == JsonValueKind.String ? dn.GetString() : internalName;
                    string? type = el.TryGetProperty("type", out var t) && (t.ValueKind == JsonValueKind.String || t.ValueKind == JsonValueKind.Number) ? t.GetRawText() : null;
                    list.Add(new SupplementaryAreasPlugin.FieldDescriptor { InternalName = internalName!, DisplayName = displayName, Type = type });
                }
            }
        }
        catch (Exception ex)
        {
            ctx.App?.Logger.LogWarning(ex, "Failed fetching active fields for record {RecordId}", recordSummary?.Id);
        }
        return list;
    }

    private List<string> BuildRequiredFieldsList()
    {
        var requiredFields = new List<string>
        {
            SupplementaryAreasPlugin.SupplementaryModel.DueDateFieldInternalName,
            SupplementaryAreasPlugin.SupplementaryModel.ExecutiveDirectorsFieldInternalName,
            SupplementaryAreasPlugin.SupplementaryModel.AllocatorsFieldInternalName,
            SupplementaryAreasPlugin.SupplementaryModel.ContributorsFieldInternalName,
        };
        
        return requiredFields;
    }

    private List<string> IdentifyMissingFields(List<string> requiredFields, List<SupplementaryAreasPlugin.FieldDescriptor> activeFields)
        => requiredFields.Where(req => !activeFields.Any(a => string.Equals(a.InternalName, req, StringComparison.OrdinalIgnoreCase))).ToList();

    private object BuildGetResponse(
        SupplementaryAreasPlugin.SupplementaryModel model,
        RecordSummary? recordSummary,
        List<SupplementaryAreasPlugin.FieldDescriptor> activeFields,
        List<string> requiredFields,
        List<string> missingFields,
        List<object> dropdownOptions)
    {
        var fields = BuildFieldsForResponse(activeFields, requiredFields);
        var missingShape = BuildMissingFieldsShape(missingFields);
        
        return new
        {
            model,
            record = recordSummary,
            fields,
            dropdownOptions,
            missingFields = missingShape,
            dueDateFieldInternalName = SupplementaryAreasPlugin.SupplementaryModel.DueDateFieldInternalName
        };
    }

    private List<object> BuildFieldsForResponse(List<SupplementaryAreasPlugin.FieldDescriptor> activeFields, List<string> requiredFields)
        => activeFields
            .Where(f => requiredFields.Contains(f.InternalName, StringComparer.OrdinalIgnoreCase))
            .Select(f => new { internalName = f.InternalName, displayName = f.DisplayName, type = f.Type, isMissing = false })
            .Cast<object>()
            .ToList();

    private object BuildMissingFieldsShape(List<string> missingFields)
        => new { global = missingFields };

    private async Task<List<object>> BuildDropdownOptionsAsync(CancellationToken ct)
    {
        if (ctx.App == null) return new List<object>();
        try
        {
            var options = await OrgLevel1OptionsService.Instance.GetAsync(ctx.App, ct).ConfigureAwait(false);
            return options.Select(o => new { id = o.id, label = o.label }).Cast<object>().ToList();
        }
        catch (OperationCanceledException) { throw; }
        catch (Exception ex)
        {
            ctx.App?.Logger.LogWarning(ex, "dropdown options failed");
            return new List<object>();
        }
    }
}

/// <summary>
/// Cached service for fetching organization level 1 dropdown options
/// </summary>
internal sealed class OrgLevel1OptionsService
{
    private static readonly TimeSpan CacheTtl = TimeSpan.FromMinutes(10);
    private (DateTime fetchedUtc, List<(string id, string label)> data)? _cache;
    private readonly SemaphoreSlim gate = new(1, 1);
    public static OrgLevel1OptionsService Instance { get; } = new();

    private OrgLevel1OptionsService() { }

    public async Task<List<(string id, string label)>> GetAsync(IAppAccess app, CancellationToken ct)
    {
        var now = DateTime.UtcNow;
        var cached = _cache;
        if (cached.HasValue && (now - cached.Value.fetchedUtc) < CacheTtl)
            return cached.Value.data;

        await gate.WaitAsync(ct).ConfigureAwait(false);
        try
        {
            // Re-check after acquiring lock
            cached = _cache;
            if (cached.HasValue && (now - cached.Value.fetchedUtc) < CacheTtl)
                return cached.Value.data;

            var fresh = await FetchAsync(app, ct).ConfigureAwait(false);
            _cache = (DateTime.UtcNow, fresh);
            return fresh;
        }
        finally { gate.Release(); }
    }

    private static async Task<List<(string id, string label)>> FetchAsync(IAppAccess app, CancellationToken ct)
    {
        var list = new List<(string id, string label)>();
        try
        {
            string json = await app.UniversalAppAccess.InvokeActionAsync(UniversalActionGetOrgLevel1, string.Empty, ct).ConfigureAwait(false);
            if (string.IsNullOrWhiteSpace(json)) return list;
            using var doc = JsonDocument.Parse(json);
            if (doc.RootElement.ValueKind != JsonValueKind.Array) return list;
            foreach (var el in doc.RootElement.EnumerateArray())
            {
                if (el.ValueKind != JsonValueKind.Object) continue;
                if (el.TryGetProperty("name", out var nameEl) && nameEl.ValueKind == JsonValueKind.String)
                {
                    var name = nameEl.GetString();
                    if (string.IsNullOrWhiteSpace(name)) continue;
                    var id = name.Trim().ToLowerInvariant().Replace(' ', '-');
                    list.Add((id, name));
                }
            }
        }
        catch (OperationCanceledException) { throw; }
        catch { }
        return list;
    }
}
